#  A multikit "invoked command" for SDP/SAP directories
#  version: 2000.12.15

# Change the following to print out any errors that we see in incoming SAP announcements
set printErrors 0

# The main routine (invoked command):
proc mk_sap_watcher {ourProgId ourInternetAddress ourPortNum ourTTL ourPublicKey args} {
    # Initialize the network:
    global ourSocket 
    if {[catch {groupsock_create $ourInternetAddress $ourPortNum $ourTTL} socket]} {
	return
    }
    set ourSocket $socket
    
    # Watch for incoming data:
    fileevent $socket readable \
	    [list mk_processSDPPacket $socket $ourProgId]
}


# Accept input from "multikit", and share it on the network.
proc multikit_input {particCode incarnation
ttl dirId progId parentId expirDate args} {
    global ourSocket
    if {![info exists ourSocket]} return
    
    set sdp {}
    switch $particCode {
	d	-
	p { # SAP doesn't support 'posting', so treat it like 'displaying'
	    set sdp [eval generateSDP $incarnation $progId $expirDate $args]
	}
	x { # Don't implement SAP deletion, because it has weak security
	}
    }		
    
    if {$sdp != {}} {
	sap_write $ourSocket $ttl $sdp
    }
    
    # Do this again after a delay:
    set dataSize [string length $sdp]
    set delayMilliseconds [computeSAPdelay $dirId $ttl $dataSize]
    after $delayMilliseconds reinput $dirId $progId
}

# Clean up state, prior to exiting.
proc multikit_cleanup {dirId} {
    global ourSocket
    groupsock_delete $ourSocket
}

proc reinput {dirId progId} {
    # First, make sure our directory is still outputting this item,
    # and, if so, get the most up-to-date version of its state.
    set currentParams [multikit_getSharingParams $dirId $progId]
    
    # Then, reinvoke the input procedure:
    if {$currentParams != {}} {
	eval multikit_input $currentParams
    }
}


# Helper routines:

proc mk_convertSDPTime {time} {
    # SDP time values have a different epoch.
    # To convert "time" to our (1970 epoch) time format, we must add
    # 0x7C558180 to it:
    # (But leave 0 as is)
    if {$time == 0} {return $time}
    
    set offset 0x7C558180
    if {[catch {expr $time+$offset} result]} {
	# overflow
	return 0x7FFFFFFF
    } else {
	return $result
    }
}

proc mk_parseTime {timeStr} {
    # Check whether $timeStr is a legal integer:
    if {![catch {expr 0+$timeStr}]} {
	# Yes, it's a legal integer.
	return [mk_convertSDPTime $timeStr]
    }
    
    # It wasn't legal	
    return 0x7FFFFFFF
}

# The opposite of the above: Convert multikit (==Unix) time to SDP time
proc mk_deParseTime {mTime} {
    if {$mTime == 0 || $mTime == 0x7FFFFFFF
    || [catch {expr $mTime - 0x7C558180} newTime]} {
	return 0
    } else {
	# Reformat "newTime" so it doesn't print negative
	set numBils 0
	while {$newTime < 0} {
	    incr newTime -1000000000
	    incr numBils
	}
	incr numBils [expr {$newTime/1000000000}]
	set newTime [expr {$newTime%1000000000}]
	if {$numBils > 0} {
	    set newTime "$numBils[format %09d $newTime]"
	} 
	return $newTime
    }
}

proc mk_parseConnectionInfo {c_field times} {
    if {$c_field != {}} {
	if {[llength $c_field] != 3 ||
	[lindex $c_field 0] != "IN" ||
	[lindex $c_field 1] != "IP4"} {
	    return {}
	}
	set groupEIdList \
		[split [lindex $c_field 2] /]
	foreach {baseAddress ttl numAddrs} $groupEIdList break
	if {$numAddrs != {}} {
	    set inetAddress $baseAddress/$numAddrs
	} else {
	    set inetAddress $baseAddress
	}
    } else {
	set inetAddress {}
	set ttl {}
    }
    
    # $times is a list of {start end repeat adjustment} records
    # (with 'repeat' and 'adjustment' being optional).
    # Convert absolute time fields to Unix time format:
    set lastStartTime [clock seconds]
    set expirationDate 0
    set newTimes {}
    foreach time $times {
	foreach {startTime endTime repeat adjustment} $time break
	set startTime [mk_parseTime $startTime]
	if {$startTime == 0x7FFFFFFF} {set startTime 0}
	if {$startTime > $lastStartTime} {set lastStartTime $startTime}
	
	set endTime [mk_parseTime $endTime]
	if {$endTime == 0} {set endTime 0x7FFFFFFF}
	
	# Sanity check:
	if {$endTime <= $startTime} continue
	
	# The largest such $endTime will be our expirationDate:
	if {$endTime > $expirationDate} {set expirationDate $endTime}
	
	set newTime [list $startTime $endTime]
	if {$repeat != {}} {
	    # There's also a 'repeat' time
	    lappend newTime $repeat
	    
	    if {$adjustment != {}} {
		# There's also an 'adjustment' field
		# convert its times too:
		set newAdjustment {}
		foreach {baseTime offset} $adjustment {
		    lappend newAdjustment \
			    [mk_parseTime $baseTime] $offset
		}
		lappend newTime $newAdjustment
	    }
	}
	
	lappend newTimes $newTime
    }
    
    # HACK: We don't like unbounded expiration dates.  If we have one,
    # change it to 30 days instead:
    if {$expirationDate == 0x7FFFFFFF} {
	set expirationDate [expr $lastStartTime+2592000]
    }
    
    return [list $inetAddress $ttl $expirationDate $newTimes]
}

proc mk_parseSSMInfo {attr} {
    if {[lindex $attr 0] == "source-filter:incl"} {
	set attr [lrange $attr 1 end]
    } elseif {[lindex $attr 0] == "source-filter:"
	&& [lindex $attr 1] == "incl"} {
        set attr [lrange $attr 2 end]
    } else {
	return {}
    }
    if {[lindex $attr 0] != "IN" ||
        [lindex $attr 1] != "IP4"} {
	return {}
    }
	
    # Note: This won't work if the multicast address
    # ([lindex $attr 2]) is a "*" wildcard
    return [list [lindex $attr 2] [lindex $attr 3]]
}

proc mk_processSDPPacket {socket ourProgId} {
    set packetData [groupsock_read $socket 8 header fromAddress]
    # The "8" is the offset of the real data within a "sd" packet.
    if {$packetData == {}} return
    
    # Begin by parsing the packet data.
    # The packet should consist of a sequence of strings, separated by \n
    # (To be safe, look also for \r, even though it's illegal)
    set packetData [split $packetData \n\r]
    if {[catch {
	set firstByte [lindex $header 0]
	# Make sure the SAP version (the first three bits) is 0 or 1
	set SAP_version [expr {$firstByte&0xD0}]
	if {$SAP_version != 0 && $SAP_version != 0x20} {
	    error "unknown SAP version: [expr {($SAP_version >> 5)&0x7}]"
	}
	
	# Check the address type bit (bit 3)
	set addrType [expr {$firstByte&0x10}]
	if {$addrType != 0} {
	    error "ignoring apparent IPv6-type SAP announcement"
	}
	
	# Bit 4 is ignored
	
	# Check the "message type" field (bit 5)
	set messageType [expr {$firstByte&0x04}]
	if {$messageType == 0} {
	    set deleting 0
	} else {
	    set deleting 1
	}
	
	# Check the encryption and compression bits (bits 6 and 7)
	set encComp [expr {$firstByte&0x03}]
	if {$encComp != 0} {
	    set packetData {}
	    error "ignoring encrypted and/or compressed SAP announcement from $fromAddress (header byte: $firstByte)"
	}
	
	set numMedia 0
	set miscAttrs(SDPversion) {}
	set times {}
	set wholeProg_connection {}
	foreach line $packetData {
	    # Each line should begin with "<char>=..." (or be empty):
	    if {$line == ""} continue
	    if {[string range $line 1 1] != "="} {
		error "packet contains bad line: $line"
	    }
	    
	    set designator [string range $line 0 0]
	    set value [string range $line 2 end]
	    switch $designator {
		v {set miscAttrs(SDPversion) $value}
		o {set o_field $value}
		s {set miscAttrs(nickname) $value}
		u {set miscAttrs(URI) $value}
		e {set miscAttrs(email_address) $value}
		p {set miscAttrs(phoneNum) $value}
		
		t {
		    if {[llength $value] != 2} {error "bad t= field: $value"}
		    lappend times $value
		}
		r {
		    if {[llength $value] < 2} {error "bad r= field: $value"}
		    # That test should really be "< 3",
		    # but we allow a NULL 'offset', meaning 0
		    set lastTime [lindex $times end]
		    if {[llength $lastTime] == 2} {
			# append $value onto the end of the last 'time'
			lappend lastTime $value
			set times [lreplace $times end end $lastTime]
		    }
		}
		z {
		    if {[llength $value]%2 != 0} {error "bad z= field: $value"}
		    set lastTime [lindex $times end]
		    if {[llength $lastTime] == 3} {
			# append $value onto the end of the last 'time'
			lappend lastTime $value
			set timesLength [llength $times]
			set times \
				[lrange $times 0 [expr $timesLength-2]]
			lappend times $lastTime
		    }
		}
		
		m {
		    set media($numMedia) $value
		    incr numMedia
		}
		
		i {
		    if {$numMedia == 0} {
			set miscAttrs(info) $value
		    } else {
			lappend media_miscAttrs([expr {$numMedia-1}]) \
				info $value
		    }
		}
		c {
		    if {$numMedia == 0} {
			set wholeProg_connection $value
		    } else {
			set connection([expr {$numMedia-1}]) $value
		    }
		}
		b {
		    if {$numMedia == 0} {
			set miscAttrs(bandwidth) $value
		    } else {
			lappend media_miscAttrs([expr {$numMedia-1}]) \
				bandwidth $value
		    }
		}
		k {
		    if {$numMedia == 0} {
			set wholeProg_key $value
		    } else {
			set key([expr {$numMedia-1}]) $value
		    }
		}
		a {
		    # Check first for the special case of a SSM source spec:
		    set ssmInfo [mk_parseSSMInfo $value]
		    if {$ssmInfo != {}} {
			set ssmSource([lindex $ssmInfo 0]) [lindex $ssmInfo 1]
		    } elseif {$numMedia == 0} {
			lappend sdp_session_attrs $value
		    } else {
			lappend sdp_media_attrs([expr {$numMedia-1}]) $value
		    }
		}
		
		default {set miscAttrs(sdp_$designator) $value}
	    }		
	}
	
	# Check for required fields:
	if {!$deleting} {
	    # Check for a valid version number:
	    if {$miscAttrs(SDPversion) != 0} {
		error "bad v= field: $miscAttrs(SDPversion)!"
	    }
	    
	    # Check for a session name:
	    if {![info exists miscAttrs(nickname)]} {
		error "bad packet - no s= field!"
	    }
	}
	
	# Check for an originator:
	if {![info exists o_field]} {
	    error "bad packet - no o= field!"
	}
	# Use this info as a unique program id:
	if {[llength $o_field] != 6} {
	    set o_fieldLength [llength $o_field]
	    if {$o_fieldLength > 6} {
		# Some SDP creators have a bug that splits the first
		# (user name) field into multiple words.  Fix this:
		set firstRecord [list [lrange $o_field 0 [expr $o_fieldLength-6]]]
		set o_field [concat $firstRecord [lrange $o_field [expr $o_fieldLength-5] end]]
	    } else {
		error "bad packet - bad o= field: $o_field"
	    }
	}
	set miscAttrs(originator) [lindex $o_field 0]
	# For now, assume no keys are used; pretend source IP addr is the key:
	set progId P:[lindex $o_field 5]:[lindex $o_field 1]
	
	# Handle a deletion request now:
	if {$deleting} {
	    multikit_removeMember $ourProgId $progId
	    return
	}
	
	# Parse the "connection" information into
	# "internetAddress ttl expirationDate sdpTimes":
	set resultInfo [mk_parseConnectionInfo $wholeProg_connection $times]
	if {$resultInfo == {}} {
	    error "bad c= field: \"$wholeProg_connection\", or times: \"$times\""
	}
	foreach {internetAddress ttl expirationDate sdpTimes} $resultInfo break
	set miscAttrs(sdpTimes) $sdpTimes
	
	# Construct the arguments to the "multikit_output" command
	# (This command will enter the new information as a 'bundle'
	#  if there's more than one medium)
	set incarnation [mk_parseTime [lindex $o_field 2]]
	lappend outputArgs "d" $incarnation $ourProgId $progId
	if {$numMedia != 1} {
	    set isBundle 1
	    lappend outputArgs R::TEMPLATE-Bundle $expirationDate bundle
	} else {
	    set isBundle 0
	}
	for {set i 0} {$i < $numMedia} {incr i} {
	    # Get the medium, port #, etc. from each "media" string:
	    set med $media($i)
	    if {[llength $med] < 4} {
		error "bad media field: \"$med\""
	    }
	    foreach {medium portNum transport} $med break
	    set formats [lrange $med 3 end]
	    
	    # Check for a directory session announcement:
	    if {([string compare $medium "application"] == 0
	    && [string compare $transport "SAP"] == 0)
	    || [string compare $medium "application/directory"] == 0} {
		set medium "directory"
	    }
	    if {[string compare $medium "directory"] == 0} {
		set template [multikit_parent $progId]
		if {$template == {}} { # program not yet known
		    set template $ourProgId
		}
		if {[string match mafp* [string tolower $formats]]} {
		    set invokedCmd mk_default_dir_sharer
		} else {
		    set invokedCmd [multikit_getAttribute $ourProgId invokedCmd]
		    if {$invokedCmd == {}} {set invokedCmd mk_sap_watcher}
		}
	    } else {
		set template R::TEMPLATE-AtomicChannel
		set invokedCmd mk_sdp_$medium
	    }
	    lappend media_miscAttrs($i) \
		    sdp_transport $transport sdp_formats $formats
	    
	    # Check whether we have medium-specific connection info:
	    if {[info exists connection($i)]} {
		set resultInfo [mk_parseConnectionInfo \
			$connection($i) $times]
		if {$resultInfo == {}} {
		    error "bad c= field: \"$connection($i)\", or times: \"$times\""
		}
		foreach {internetAddress1 ttl1 expirationDate1 sdpTimes1} $resultInfo break
		lappend media_miscAttrs($i) sdpTimes $sdpTimes1
	    } else {
		if {$internetAddress == {}} {
		    error "no c= field seen for medium $medium"
		}
		set internetAddress1 $internetAddress
		set ttl1 $ttl
		set expirationDate1 $expirationDate
	    }
	    if {[info exists ssmSource($internetAddress1)]} {
		set source $ssmSource($internetAddress1)
		set internetAddress1 [list $internetAddress1 $source]
	    }
	    
	    # Check whether an encryption key was specified:
	    if {[info exists key($i)] && $key($i) != ""} {
		set key1 $key($i)
	    } elseif {[info exists wholeProg_key] && $wholeProg_key != ""} {
		set key1 $wholeProg_key
	    } else {
		set key1 nokey
	    }
	    
	    if {$isBundle} {
		if {$i > 0} {
		    lappend outputArgs "|"
		}
		lappend outputArgs $progId:$medium-$portNum
		set nickname $miscAttrs(nickname):$medium
	    } else {
		set nickname $miscAttrs(nickname)
	    }
	    lappend outputArgs $template \
		    $expirationDate1 channel $internetAddress1 \
		    $portNum $ttl1 $key1
	    # Now, append any attributes that are specific to this entry:
	    lappend outputArgs invokedCmd $invokedCmd nickname $nickname
	    if {[info exists media_miscAttrs($i)]} {
		eval lappend outputArgs $media_miscAttrs($i)
	    }
	    if {[info exists sdp_media_attrs($i)]} {
		lappend outputArgs sdp_media_attributes $sdp_media_attrs($i)
	    }
	}
	
	# Finally, append attributes that apply to the whole bundle,
	# and evaluate the command
	if {$isBundle} {
	    lappend outputArgs "|"
	}
	if {[info exists sdp_session_attrs]} {
	    set ssa [list sdp_session_attributes $sdp_session_attrs]
	} else {
	    set ssa {}
	}
	
	eval multikit_output $outputArgs $ssa [array get miscAttrs]
    } errMsg]} {
	catch {
	    global printErrors
	    if {$printErrors} {
		puts stderr "mk_processSDPPacket: $errMsg"
		if {$packetData != {}} {
		    puts stderr "packetData (from $fromAddress): $packetData; 8-byte header:$header"
		}
	    }
	}
	return
    }
}


proc generateSDP {incarnation progId expirDate progType args} {
    # Begin by flattening the program's 'bundle' structure (if any),
    # to produce a list of subsession descriptions, and a list of
    # global attributes:
    set subsessions {}
    switch $progType {
	"channel" {
	    foreach {addr port ttl key} $args break
	    set args [lrange $args 4 end]
	    array set attr $args ;# global attributes
	    lappend subsessions $addr $port $ttl $key {}
	}
	bundle {
	    # $args: progId' parentId' expir' progType' args' "|" ...
	    while {[set endMarkerPosn [lsearch -exact $args "|"]] != -1} {
		if {$endMarkerPosn > 0} {
		    set progType1 [lindex $args 3]
		    # Later allow sub-bundles too
		    if {[string compare $progType1 channel] == 0} {
			set args1 [lrange $args 4 [expr {$endMarkerPosn-1}]]
			foreach {addr port ttl key} $args1 break
			set localAttrs [lrange $args1 4 end]
			lappend subsessions $addr $port $ttl $key $localAttrs
		    }
		}
		set args [lrange $args [expr {$endMarkerPosn+1}] end]
	    }
	    array set attr $args ;# global attributes
	}
	default {
	    array set attr $args ;# global attributes
	}
    }
    
    set sdp {}
    # Check for each SDP-relevant attribute:
    catch {
	# v=
	append sdp v=0\n
	
	# o=
	# Extract the (creator's) address and session id from "progId":
	foreach {prot address sessionId} [split $progId :] break
	set orig - ; catch {set orig $attr(originator)}
	set o "$orig $sessionId [mk_deParseTime $incarnation] IN IP4 $address"
	append sdp o=$o\n
	
	# s=
	append sdp s=$attr(nickname)\n
	
	# Optional global attributes:
	foreach tag {i u e p b} \
		aName {info URI email_address phoneNum bandwidth} {
	    catch {append sdp $tag=$attr($aName)\n}
	}
	
	# Note: We set "c=" and "k=" at the media level only
	
	# Get the list of time info, for the t=, r=, and z= tags:
	set times {}
	catch {set times $attr(sdpTimes)}
	if {$times == {}} {
	    set startTime {}
	    catch {set startTime $attr(startTime)}
	    if {$startTime == {}} {
		set startTime 0
	    }
	    set times [list [list $startTime $expirDate]]			
	}
	
	# t=, r=
	set z {}
	foreach time $times {
	    foreach {start end rpt adjs} $time break
	    if {$end > $expirDate} {set end $expirDate}
	    if {$start > $end} continue
	    append sdp "t=[mk_deParseTime $start] [mk_deParseTime $end]\n"
	    if {$rpt != {}} {
		append sdp "r=$rpt\n"
	    }
	    if {$adjs != {}} {
		foreach {adjTime offset} $adjs {
		    lappend z [mk_deParseTime $adjTime] $offset
		}
	    }
	}
	# z= (optional)
	if {$z != {}} {
	    append sdp z=$z\n
	}
	
	# a= (optional)
	catch {
	    foreach a $attr(sdp_session_attributes) {
		if {[string match tool:* $a]} continue
		append sdp a=$a\n
	    }
	}
	append sdp "a=tool:[multikit_getNameAndVersion]\n"
	
	# Now, fill in the SDP fields for each media 'subsession':
	foreach {addr port ttl key ssArgs} $subsessions {
	    catch {unset ssAttr}
	    array set ssAttr $ssArgs
	    catch {unset ssAttrAll}
	    array set ssAttrAll [array get attr]
	    array set ssAttrAll $ssArgs
	    
	    # m=
	    set ic {}
	    catch {set ic $ssAttr(invokedCmd)}
	    if {$ic == {}} {
		set ic [multikit_getAttribute $progId invokedCmd]
	    }
	    switch -glob $ic {
		mk_sdp_* {
		    set medium [string range $ic 7 end]
		    set transport $ssAttrAll(sdp_transport)
		    set format $ssAttrAll(sdp_formats)
		}
		mk_sap_watcher {
		    set medium application
		    set transport SAP
		    set format SDP
		}
		mk_default_dir_sharer {
		    set medium directory
		    set transport UDP
		    set format MAFP
		}
		default {error "unknown \"invokedCmd\" ($ic) for SDP"}
	    }
	    append sdp "m=$medium $port $transport $format\n"
	    
	    # i= (optional)
	    catch {append sdp i=$ssAttr(info)\n}
	    
	    # c=
	    foreach {baseAddr numAddrs} [split $addr /] break
	    set sdpAddr $addr/$ttl
	    if {$numAddrs != {}} {
		append sdpAddr /$numAddrs
	    }
	    append sdp "c=IN IP4 $sdpAddr\n"
	    
	    # b= (optional)
	    catch {append sdp b=$ssAttr(bandwidth)\n}
	    
	    # k= (optional)
	    if {$key != {} && [string compare $key nokey] != 0} {
		append sdp k=$key\n
	    }
	    
	    # a= (optional)
	    catch {
		foreach a $ssAttrAll(sdp_media_attributes) {
		    append sdp a=$a\n
		}
	    }
	}
    }
    
    return $sdp
}

proc numAnnouncers {dirId} {
    global dirSize
    if {[catch {set size $dirSize($dirId)}] || $size < 0} {
	# We need to get an updated size
	set size [llength [multikit_listAllMembers $dirId]]
	set dirSize($dirId) $size
	
	# Use this size for the next minute
	after 60000 [list set dirSize($dirId) -1]
    }
    return $size
}

proc computeSAPdelay {dirId ttl dataSize} {
    # Get the overall b/w (in bytes/sec) for this TTL:
    if {$ttl < 16} {
	set limit 1250
    } elseif {$ttl < 64} {
	set limit 250
    } elseif {$ttl < 128} {
	set limit 125
    } else {
	set limit 25
    }
    
    # Figure out the delay that will give us our share:
    set delay [expr {([numAnnouncers $dirId]*$dataSize)/$limit}]
    
    # However, impose a minimum wait of 10s, and a maximum of 20 min
    if {$delay < 10} {
	set delay 10
    } elseif {$delay > 1200} {
	set delay 1200
    }
    
    # Choose a random time, in milliseconds, between 2/3*$delay and 4/3*$delay:
    set delay [randomBetween [expr {$delay*667}] [expr {$delay*1333}]]
    return $delay
}

# Initially, compute the 8-byte SAP prefix that we use on outgoing data
set sapPrefix {0x20 0x00 0x00 0x00}
set ourSourceAddr [split [groupsock_getOurIPAddress] .]
foreach decimalByte $ourSourceAddr {
    lappend sapPrefix [format 0x%x $decimalByte]
}
proc sap_write {socket ttl data} {
    global sapPrefix
    groupsock_write -no0 $socket $data $ttl 8 $sapPrefix
}






